﻿<#
.DESCRIPTION
    This script guides a user through the the non-Vault portions of Collaborative Authoring setup
    for configurations that do not use a Vault Collaboration User.
#>

$version = "1.0"

<#
.DESCRIPTION
    Helper function for yes/no responses.
#>
function Get-YesNoResponse {
    param (
        [Parameter(Mandatory=$true)]
        [string]$prompt
    )

    $response = ""
    while ($true) {
        Write-Host -f Yellow "$prompt (Y/N)"
        $response = Read-Host
        Write-Host "Response entered: $response"
        if ($response -match "^[Yy]$") {
            return $true
        } elseif ($response -match "^[Nn]$") {
            return $false
        } else {
            Write-Warning "Invalid input. Please enter 'Y' for Yes or 'N' for No."
        }
    }
}

<#
.DESCRIPTION
    Helper function for to prompt user for non-empty string.
#>

function Get-NonEmptyString {
    param(
        [Parameter(Mandatory = $true)]
        [string]$prompt
    )

    do {
        Write-Host -f Yellow $prompt
        $response = Read-Host

        if ([string]::IsNullOrEmpty($response)) {
            Write-Warning "Input cannot be empty. Please try again."
        }
    } while ([string]::IsNullOrEmpty($response))

    Write-Host "Response entered: $response"
    return $response
}

<#
.DESCRIPTION
    Helper function to prompt user for a valid site alias.
#>

function Get-SiteAlias {
    $isValid = $false
    do {
        Write-Host -f Yellow "`nPlease enter your new site's alias."
        $alias = Read-Host
        Write-Host "Response entered: $alias"

        if ([string]::IsNullOrEmpty($alias)) {
            Write-Warning "Input cannot be empty. Please try again."
        } elseif ($alias -like '* *') {
            Write-Warning "Alias cannot contain spaces. Please try again."
        } elseif ($alias -match "[^a-zA-Z0-9._-]") {
            Write-Warning "Alias contains invalid symbols. Only letters, numbers, periods (.), underscores (_), and dashes (-) are allowed. Please try again."
        } elseif ($alias -like '_*') {
            Write-Warning "Alias cannot start with an underscore (_). Please try again."
        } elseif ($alias -like '.*') {
            Write-Warning "Alias cannot start with a period (.). Please try again."
        } elseif ($alias -like '*.') {
            Write-Warning "Alias cannot end with a period (.). Please try again."
        } else {
            $isValid = $true
        }
    } while (-not $isValid)

    return $alias
}

<#
.DESCRIPTION
    Helper function to check if script was run with compatible Powershell and PnP Powershell versions.
#>

function Test-RequiredVersions {
    Write-Host -f Cyan "`nChecking for required software versions..."
    $allChecksPassed = $true

    Write-Host "`n--- PowerShell Version Check ---"
    $requiredPSVersion = [System.Version]'7.0.0'
    $installedPSVersion = $PSVersionTable.PSVersion
    Write-Host "Required minimum version: $($requiredPSVersion.ToString())"
    Write-Host "Your installed version:   $($installedPSVersion.ToString())"

    if ($installedPSVersion.Major -ge $requiredPSVersion.Major) {
        Write-Host -f Green "✅ PowerShell version requirement met."
    } else {
        Write-Error "Your PowerShell version is too old. Please upgrade to version 7.0 or newer."
        $allChecksPassed = $false
    }

    Write-Host "`n--- PnP PowerShell Version Check ---"
    $requiredPnPVersion = [System.Version]'3.1.0'
    $installedPnPModule = Get-Module -Name "PnP.PowerShell" -ListAvailable | Sort-Object -Property Version -Descending | Select-Object -First 1
    Write-Host "Required minimum version: $($requiredPnPVersion.ToString())"

    if (-not $installedPnPModule) {
        Write-Error "PnP.PowerShell module is not installed. Please install version 3.1.0 or newer."
        $allChecksPassed = $false
    } else {
        $installedPnPVersion = $installedPnPModule.Version
        Write-Host "Your installed version:   $($installedPnPVersion.ToString())"

        if ($installedPnPVersion -ge $requiredPnPVersion) {
            Write-Host -f Green "✅ PnP.PowerShell version requirement met."
        } else {
            Write-Error "Your PnP.PowerShell module is too old. Please update to version 3.1.0 or newer."
            $allChecksPassed = $false
        }
    }

    return $allChecksPassed
}

<#
.DESCRIPTION
    Helper function to get the client id from a Register-PnPEntraIDAppForInteractiveLogin response.
#>
function Get-ClientID {
    param(
        [Parameter(Mandatory = $true)]
        [Object]$response
    )

    $clientId = $null
    $regexMatch = $response | Select-String -Pattern "@{AzureAppId/ClientId=" -AllMatches
    if ($regexMatch) {
        $clientId = $regexMatch -replace ".*=" -replace "}"
    } else {
        Write-Error "Failed to get client id"
    }

    return $clientId
}

# Main script start

$logFileName = "./Collab-Auth-Setup-Log-$(Get-Date -Format 'yyyy-MM-dd_HH-mm-ss').txt"
Write-Host -f Yellow "`nSession logging will now start. Output will be saved to: $logFileName."
Start-Transcript -Path $logFileName
Write-Host -f Gray "`nRunning script version: $version"

$shouldContinue = Test-RequiredVersions
if ($shouldContinue -eq $false) {
    Write-Host -f Green "`nScript finished executing."
    exit
}

$shouldContinue = Get-YesNoResponse -Prompt "`nPlease confirm you have read the README and are ok with granting the permissions necessary to run this script."
if ($shouldContinue -eq $false) {
    Write-Host -f Green "`nScript finished executing."
    exit
}

Write-Host -f Cyan "`nCreating temporary Entra ID admin app..."

$tenantDomain = Get-NonEmptyString -Prompt "`nPlease enter your tenant domain (e.g. veeva.onmicrosoft.com)."
$tempAdminAppName = "TempVeevaVaultAdminApp"
$tempAdminAppClientId = $null
try {
    Write-Host "`nYou are about to be prompted to log in and then prompted again for consent. Please log in as an admin account."
    Write-Host "When prompted to consent, please ensure you check the box to consent on behalf of your organization."
    Read-Host -Prompt "Press enter to continue"

    $response = Register-PnPEntraIDAppForInteractiveLogin -ApplicationName $tempAdminAppName -SharePointDelegatePermissions "AllSites.FullControl" -GraphDelegatePermissions "Application.ReadWrite.All", "Group.ReadWrite.All" -Tenant $tenantDomain
    $tempAdminAppClientId = Get-ClientID -response $response

    if ($tempAdminAppClientId) {
        Write-Host -f Green "Created Entra ID App with client id: $tempAdminAppClientId"                
    } else {
        $shouldContinue = $false
    }
} catch {
    if ($_.FullyQualifiedErrorId -match "InvalidOperation") {
        Write-Host -f Yellow "`nEntra ID app with name '$tempAdminAppName' may already exist. Please delete that app and restart this script or provide the app's client id."
        $tempAdminAppClientId = Get-NonEmptyString -Prompt "Please provide the client id for '$tempAdminAppName'"
    } else {
        Write-Error "Failed to create admin app: $_"
        Write-Host -f Green "`nScript finished executing."
        exit
    }
}

if ($shouldContinue) {
    $sitesSelectedAppClientId = $null
    $userHasSitesSelectedApp = Get-YesNoResponse -Prompt "`nDo you already have an Entra ID app that will be used for the Collaborative Authoring configuration?"
    if ($userHasSitesSelectedApp) {
        Write-Host @"
`nPlease ensure that your existing Entra ID app has the Microsoft Graph 'Sites.Selected' application permission.
Additionally, the following optional features require additional permissions:
    - Automatic external user invites: Directory.ReadWrite.All & User.Invite.All application permissions
    - Automatic adding of participants during workflows: User.Read.All application permission
"@
        $sitesSelectedAppClientId = Get-NonEmptyString -Prompt "`nPlease enter the client ID of your existing Entra ID app."
    } else {
        Write-Host -f Cyan "`nCreating Entra ID app for Collaborative Authoring configuration..."

        $sitesSelectedAppName = Get-NonEmptyString -Prompt "`nPlease enter the name you want the app to have."
        $graphPermissions = @("Sites.Selected")

        $shouldAddInvitePermissions = Get-YesNoResponse -Prompt "`nDo you want to add the permissions necessary for automatic external user invites (Directory.ReadWrite.All & User.Invite.All)?"
        if ($shouldAddInvitePermissions) {
            $graphPermissions += @(
                "Directory.ReadWrite.All",
                "User.Invite.All"
            )
        }

        $shouldAddMentioningPermission = Get-YesNoResponse -Prompt "`nDo you want to add the permission necessary for automatic adding of participants during workflows (User.Read.All)?"
        if ($shouldAddMentioningPermission) {
            $graphPermissions += "User.Read.All"
        }

        try {
            Write-Host "`nYou are about to be prompted to log in and then prompted again for consent. Please log in as an admin account."
            Read-Host -Prompt "Press enter to continue"

            $response = Register-PnPEntraIDAppForInteractiveLogin -ApplicationName $sitesSelectedAppName -GraphApplicationPermissions $graphPermissions -Tenant $tenantDomain
            $sitesSelectedAppClientId = Get-ClientID -response $response

            if ($sitesSelectedAppClientId) {
                Write-Host -f Green "Created Entra ID App with client id: $sitesSelectedAppClientId"                
            } else {
                $shouldContinue = $false
            }
        } catch {
            Write-Error "Failed to create Collaborative Authoring app: $_"
            $shouldContinue = $false
        }
    }
}

$sharePointAdminConnection = $null
$sharePointAdminUrl = Get-NonEmptyString -Prompt "`nPlease enter your SharePoint admin URL (e.g. https://veeva-admin.sharepoint.com/)."
Read-Host -Prompt "`nYou are about to be prompted to log in. Please log in as an admin account. Press enter to continue"
try {
    $sharePointAdminConnection = Connect-PnPOnline -Url $sharePointAdminUrl -ClientId $tempAdminAppClientId -Interactive -ReturnConnection
    Write-Host -f Yellow "Waiting 10 seconds for connection to be established..."
    Start-Sleep -Seconds 10
} catch {
    Write-Error "Failed to login as SharePoint admin: $_"
    Write-Error "We are unable to delete the '$tempAdminAppName' Entra ID app without a SharePoint admin connection. Please delete app manually."
    exit
}

if ($shouldContinue) {
    $sharePointSiteUrl = $null
    $userHasSharePointSite = Get-YesNoResponse -Prompt "`nDo you already have a SharePoint site?"
    if ($userHasSharePointSite) {
        $sharePointSiteUrl = Get-NonEmptyString -Prompt "`nPlease enter your SharePoint site URL (e.g. https://veeva.sharepoint.com/sites/MySite)."
    } else {
        Write-Host -f Cyan "`nCreating SharePoint site..."

        Write-Host "`nYour new SharePoint site will need a name and an alias. The site's alias is what shows up in the site's URL (e.g. MySiteAlias -> https://veeva.sharepoint.com/sites/MySiteAlias)."
        $siteTitle = Get-NonEmptyString -Prompt "`nPlease enter your new site's name."
        $siteAlias = Get-SiteAlias

        try {
            $sharePointSiteUrl = New-PnPSite -Type TeamSite -Title $siteTitle -Alias $siteAlias -Connection $sharePointAdminConnection
            Write-Host -f Green "`nSharePoint site created: $sharePointSiteUrl"
        } catch {
            Write-Error "Failed to create SharePoint site: $_"
            $shouldContinue = $false
        }
    }
}

if ($shouldContinue) {
    $shouldHardenSiteSecurity = Get-YesNoResponse -Prompt "`nDo you need your SharePoint site's security hardened?"
    if ($shouldHardenSiteSecurity) {
        Write-Host -f Cyan "`nSecurity hardening SharePoint site..."

        $documentsLibraryName = "Documents"
        $usingCustomDocumentLibrary = Get-YesNoResponse -Prompt "`nAre you using a custom document library for your site? (Note: Most customers use the default document library)"
        if ($usingCustomDocumentLibrary) {
            $documentsLibraryName = Get-NonEmptyString -Prompt "`nPlease enter your custom document library name."
        }

        $sharePointSiteConnection = $null
        try {
            Read-Host -Prompt "`nYou are about to be prompted to log in. Please log in as an admin account. Press enter to continue"
            $sharePointSiteConnection = Connect-PnPOnline -Url $sharePointSiteUrl -ClientId $tempAdminAppClientId -Interactive -ReturnConnection
            Write-Host -f Yellow "Waiting 10 seconds for connection to be established...`n"
            Start-Sleep -Seconds 10
        } catch {
            Write-Error "Failed to login to SharePoint site: $_"
            Write-Error "Security hardening will be skipped. You will need to re-run the script or manually change security settings."
            $shouldHardenSiteSecurity = $false
        }

        if ($shouldHardenSiteSecurity) {
            try {
                Set-PnPWeb -MembersCanShare:$false -Connection $sharePointSiteConnection
                Write-Host -f Green "Member sharing disabled."
            } catch {
                Write-Error "Error disabling member sharing: $_"
            }

            try {
                Set-PnPRequestAccessEmails -Disabled -Connection $sharePointSiteConnection
                Write-Host -f Green "Access requests disabled."
            } catch {
                Write-Error "Error disabling access requests: $_"
            }

            try {
                Set-PnPRoleDefinition -Identity "Contribute" -Clear DeleteListItems -Connection $sharePointSiteConnection | Out-Null
                Write-Host -f Green "Deleting files disabled for the Contribute permission level."
            } catch {
                Write-Error "Error disabling deleting files for Contribute permission level: $_"
            }

            try {
                $list = Get-PnPList -Identity $documentsLibraryName -Includes HasUniqueRoleAssignments -Connection $sharePointSiteConnection
                if (-not $list.HasUniqueRoleAssignments) {
                    Set-PnPList -Identity $list.Title -BreakRoleInheritance -CopyRoleAssignments -Connection $sharePointSiteConnection | Out-Null
                    Write-Host -f Green "Permission inheritance on '$documentsLibraryName' library has been broken."
                } else {
                    Write-Host -f Yellow "The '$documentsLibraryName' library is already using unique permissions."
                }
            } catch {
                Write-Error "Error breaking permission inheritance on '$documentsLibraryName': $_"
            }

            try {
                $context = Get-PnPContext -Connection $sharePointSiteConnection
                $allGroups = Get-PnPGroup -Connection $sharePointSiteConnection
                ForEach($group in $allGroups) {
                    $groupTitle = $group.Title
                    if ($groupTitle -match "members|visitors") {
                        $list.RoleAssignments.GetByPrincipalId($group.Id).DeleteObject()
                        try {
                            $context.ExecuteQuery()
                            Write-Host -f Green "Group '$groupTitle' removed."
                        } catch {
                            Write-Host -f Yellow "Error removing group '$groupTitle'. Group may have already been removed."
                        }
                    }
                }
            } catch {
                Write-Error "Error fetching site groups: $_"
            }
        }
    }
}

if ($shouldContinue) {
    try {
        Write-Host -f Cyan "`nGranting Collaborative Authoring Entra ID app permission to the SharePoint site..."

        Grant-PnPAzureADAppSitePermission -AppId $sitesSelectedAppClientId -DisplayName "Collaborative Authoring" -Site $sharePointSiteUrl -Permissions Write -Connection $sharePointAdminConnection
        Write-Host -f Green "Granted permission to site"
    } catch {
        Write-Error "Failed to grant permission to site: $_"
    }
}

Write-Host -f Cyan "`nDeleting temporary Entra ID admin app..."
Start-Sleep -Seconds 10
try {
    Remove-PnPEntraIDApp -Identity $tempAdminAppName -Connection $sharePointAdminConnection -Force
    Write-Host -f Green "Temporary admin app deleted."             
} catch {
    Write-Error "`nUnable to delete temporary admin app: $tempAdminAppName. Please delete app manually."
}

Write-Host -f Green "`nScript finished executing."
Stop-Transcript