﻿# Prerequisites:
# 1. Install SharePoint Online Management Shell
# Install-Module -Name Microsoft.Online.SharePoint.PowerShell
# 2. Ensure you have appropriate permissions to create sites
Add-Type -Path "C:\Program Files\Common Files\Microsoft Shared\Web Server Extensions\16\ISAPI\Microsoft.SharePoint.Client.dll"
Add-Type -Path "C:\Program Files\Common Files\Microsoft Shared\Web Server Extensions\16\ISAPI\Microsoft.SharePoint.Client.Runtime.dll"

# Function to security harden a site
Function Setup-SharePointSite($siteUrl, $siteTitle) {
    # Setup Credentials to connect
    $Cred = Get-Credential
    $Cred = New-Object Microsoft.SharePoint.Client.SharePointOnlineCredentials($Cred.UserName,$Cred.Password)
    $Ctx = New-Object Microsoft.SharePoint.Client.ClientContext($siteUrl)
    $Ctx.Credentials = $Cred

    $Web = $Ctx.Web
    $AllProperties=$Web.AllProperties
    $Ctx.Load($Web)
    $Ctx.Load($AllProperties)
    $Ctx.ExecuteQuery()

    # Update Site permissions - Only site owners can share files, folders, and the site
    try {
        $Web.MembersCanShare=$false
        $web.Update();
        $web.Context.ExecuteQuery()

        Write-host -f Green "Updated site permissions so that only site owners can share files, folders, and the site"
    } catch {
        Write-Host "Error:"$_.Exception.Message -f Red 
    }

    # Disable Allow Access Requests
    try {
        $web.RequestAccessEmail = ""
        $web.Update()
        $Ctx.ExecuteQuery() 

        Write-host -f Green "Updated site permissions to disable Allow Access requests"
    } catch {
        Write-Host "Error:"$_.Exception.Message -f Red 
    }

    # Remove the "Delete Items" permission from the "Contribute" Permission Level
    # Get the role definition by name
    $PermissionLevelName = "Contribute"
    $RoleDefinition = $Ctx.web.RoleDefinitions.GetByName($PermissionLevelName)
    $Ctx.Load($RoleDefinition)
    $Ctx.ExecuteQuery()

    # Remove "Delete Items" Permission from the Permission Level
    try {
        $BasePermissions = New-Object Microsoft.SharePoint.Client.BasePermissions
        $BasePermissions = $RoleDefinition.BasePermissions
        $BasePermissions.Clear([Microsoft.SharePoint.Client.PermissionKind]::DeleteListItems)
        $RoleDefinition.BasePermissions =  $BasePermissions
        $RoleDefinition.Update()
        $Ctx.ExecuteQuery()   
        Write-host -f Green "Removed 'Delete Items' from Contribute Permission Level"
    } catch {
        Write-Host "Error:"$_.Exception.Message -f Red 
    }
     
    # Get the "Documents" list
    $ListName = "Documents"
    $List=$Ctx.web.Lists.GetByTitle($ListName)
    $Ctx.load($List)
    $List.Retrieve("HasUniqueRoleAssignments")
    $Ctx.ExecuteQuery()
 
    # Check if list is inheriting permissions; Break permissions of the list, if its inherited
    if ($List.HasUniqueRoleAssignments -eq $False)
    {
        # sharepoint online break inheritance powershell 
        $List.BreakRoleInheritance($True,$True) #keep existing list permissions & Item level permissions
        $Ctx.ExecuteQuery()
        Write-host -f Green "Permission inheritance broken successfully!"
    }
    else
    {
        Write-Host -f Yellow "List is already using Unique permissions!"
    }

    #Get List Permissions
    $Ctx.Load($List.RoleAssignments)
    $ctx.ExecuteQuery()

    $GroupNameMembers = "Members"
    $GroupNameVisitors = "Visitors"

    # Get the Members Group
    try {
        $MembersGroupName = 'Site ' + $GroupNameMembers
        $MembersGroup = $Ctx.Web.SiteGroups.GetByName($MembersGroupName) 
        $Ctx.Load($MembersGroup)
        $Ctx.ExecuteQuery()
    } catch {
        $MembersGroupName = $siteTitle + ' ' + $GroupNameMembers
        $MembersGroup = $Ctx.Web.SiteGroups.GetByName($MembersGroupName) 
        $Ctx.Load($MembersGroup)
        $Ctx.ExecuteQuery()
    }


    #Remove Group from List Permissions
    try {
        $List.RoleAssignments.GetByPrincipal($MembersGroup).DeleteObject()
        $Ctx.ExecuteQuery()
        Write-host -f Green "Removed permission from Members Group!"
    } catch {
        Write-Host "Error:"$_.Exception.Message -f Red 
    }
    
    # Get the Visitors Group
    try {
        $VisitorsGroupName = 'Site ' + $GroupNameVisitors
        $VisitorsGroup = $Ctx.Web.SiteGroups.GetByName($VisitorsGroupName) 
        $Ctx.Load($VisitorsGroup)
        $Ctx.ExecuteQuery()
    } catch {
        $VisitorsGroupName = $siteTitle + ' ' + $GroupNameVisitors
        $VisitorsGroup = $Ctx.Web.SiteGroups.GetByName($VisitorsGroupName) 
        $Ctx.Load($VisitorsGroup)
        $Ctx.ExecuteQuery()
    }

    #Remove Group from List Permissions
    try {
        $List.RoleAssignments.GetByPrincipal($VisitorsGroup).DeleteObject()
        $Ctx.ExecuteQuery()
        Write-host -f Green "Removed permission from Visitors Group!"
    } catch {
        Write-Host "Error:"$_.Exception.Message -f Red 
    }
}


# Example Usage
# Modify these parameters according to your environment
$siteUrl = "https://devopsveeva.sharepoint.com/sites/testSite2025_3"
$siteTitle = "Test Site 2025 3"
Setup-SharePointSite -siteUrl $siteUrl -siteTitle $siteTitle